<?php
/**
 * Plugin Name: WP Optimize Core
 * Plugin URI: https://example.com/wp-optimize-core
 * Description: Минималистичный плагин для отключения ненужных функций WordPress и повышения производительности. Включает отчёт о «тяжести» wp_head и автозагрузке, экспорт/импорт настроек и 16 оптимизаций.
 * Version: v1.7.1
 * Author: Optimizer
 * Author URI: https://example.com
 * License: GPL-2.0+
 * Text Domain: wp-optimize-core
 */

if (!defined('ABSPATH')) {
    exit;
}

define('WPOC_PATH', plugin_dir_path(__FILE__));
define('WPOC_URL', plugin_dir_url(__FILE__));

// Применяем оптимизации на раннем хуке
add_action('plugins_loaded', 'wpoc_apply_optimizations', 1);

function wpoc_apply_optimizations() {
    // Загружаем настройки с миграцией
    $saved_settings = get_option('wpoc_settings', []);
    $default_settings = wpoc_get_default_settings();

    $needs_update = false;
    foreach ($default_settings as $key => $default_value) {
        if (!isset($saved_settings[$key])) {
            $saved_settings[$key] = $default_value;
            $needs_update = true;
        }
    }

    if ($needs_update) {
        update_option('wpoc_settings', $saved_settings);
    }

    $settings = $saved_settings;

    // === Старые оптимизации ===
    if (!empty($settings['disable_emoji'])) {
        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_action('admin_print_styles', 'print_emoji_styles');
    }

	if (!empty($settings['block_author_query_redirect'])) {
		add_action('template_redirect', 'wpoc_block_author_query_redirect', 1);
	}

	if (!empty($settings['disable_comment_website_field'])) {
		add_filter('comment_form_default_fields', 'wpoc_remove_comment_website_field');
		add_action('preprocess_comment', 'wpoc_force_empty_comment_url', 1);
	}
	
	if (!empty($settings['canonical_pagination_to_main'])) {
		add_action('wp_head', 'wpoc_override_yoast_canonical_on_pagination', 9);
	}

    if (!empty($settings['disable_embed'])) {
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
        remove_filter('oembed_dataparse', 'wp_filter_oembed_result', 10);
    }

    if (!empty($settings['disable_xmlrpc'])) {
        add_filter('xmlrpc_enabled', '__return_false');
        add_filter('pings_open', '__return_false', 999);
    }

    if (!empty($settings['disable_rest_for_guests'])) {
        add_filter('rest_authentication_errors', 'wpoc_disable_rest_for_guests');
    }

    if (!empty($settings['reduce_heartbeat'])) {
        add_filter('heartbeat_settings', 'wpoc_reduce_heartbeat_interval');
    }

    if (!empty($settings['clean_wp_head'])) {
        wpoc_clean_wp_head();
    }

    if (!empty($settings['disable_gutenberg_frontend'])) {
        add_action('wp_enqueue_scripts', 'wpoc_dequeue_gutenberg_front', 100);
    }

    if (!empty($settings['disable_self_pingbacks'])) {
        add_action('pre_ping', 'wpoc_disable_self_pings');
    }

    if (!empty($settings['disable_rss'])) {
        add_action('init', 'wpoc_disable_rss');
    }

    if (!empty($settings['disable_author_archives'])) {
        add_action('template_redirect', 'wpoc_disable_author_archives');
    }

    if (!empty($settings['disable_jquery_migrate'])) {
        add_filter('wp_default_scripts', 'wpoc_remove_jquery_migrate');
    }

    if (!empty($settings['hide_wp_version'])) {
        add_filter('the_generator', '__return_empty_string');
        add_filter('style_loader_src', 'wpoc_remove_script_version', 9999);
        add_filter('script_loader_src', 'wpoc_remove_script_version', 9999);
    }

    if (!empty($settings['disable_wpautop'])) {
        remove_filter('the_content', 'wpautop');
        remove_filter('the_excerpt', 'wpautop');
    }

    if (!empty($settings['disable_core_updates'])) {
        add_filter('automatic_updater_disabled', '__return_true');
        add_filter('auto_update_core_dev', '__return_false');
        add_filter('auto_update_core_minor', '__return_false');
        add_filter('auto_update_core_major', '__return_false');
        add_filter('auto_update_plugin', '__return_false');
        add_filter('auto_update_theme', '__return_false');
        add_filter('wp_version_check', '__return_null');
        add_filter('wp_update_plugins', '__return_null');
        add_filter('wp_update_themes', '__return_null');
    }

    // === НОВАЯ оптимизация ===
    if (!empty($settings['set_last_modified'])) {
        add_action('template_redirect', 'wpoc_send_last_modified_header');
    }
}

// --- Функции оптимизаций ---

/**
 * Блокирует редирект с ?author=N на /author/имя/
 */
function wpoc_block_author_query_redirect() {
    // Проверяем, если запрос был через ?author=N
    if (isset($_GET['author']) && is_numeric($_GET['author'])) {
        // Отключаем редирект, который делает WordPress
        remove_action('template_redirect', 'redirect_canonical');
        // Возвращаем 404
        global $wp_query;
        $wp_query->set_404();
        status_header(404);
        nocache_headers();
        include get_404_template();
        exit;
    }
}


/**
 * Заменяет canonical на страницах пагинации на основную страницу рубрики/архива
 */
function wpoc_override_yoast_canonical_on_pagination() {
    if (is_paged()) {
        $canonical_url = '';

        if (is_home()) {
            $canonical_url = home_url('/');
        } elseif (is_category()) {
            $canonical_url = get_category_link(get_queried_object_id());
        } elseif (is_tag()) {
            $canonical_url = get_tag_link(get_queried_object_id());
        } elseif (is_tax()) {
            $canonical_url = get_term_link(get_queried_object_id());
        } elseif (is_author()) {
            $canonical_url = get_author_posts_url(get_queried_object_id());
        } elseif (is_date()) {
            $canonical_url = get_month_link(get_query_var('year'), get_query_var('monthnum'));
        } elseif (is_post_type_archive()) {
            $canonical_url = get_post_type_archive_link(get_post_type());
        }

        if ($canonical_url) {
            $canonical_url = user_trailingslashit($canonical_url);
            echo '<link rel="canonical" href="' . esc_url($canonical_url) . '" />' . "\n";
        }
    }
}


/**
 * Убирает поле "Сайт" из формы комментариев
 */
function wpoc_remove_comment_website_field($fields) {
    unset($fields['url']);
    return $fields;
}

/**
 * Принудительно обнуляет URL в комментарии (на всякий случай)
 */
function wpoc_force_empty_comment_url($comment_data) {
    $comment_data['comment_author_url'] = '';
    return $comment_data;
}


function wpoc_disable_rest_for_guests($result) {
    if (!empty($result)) {
        return $result;
    }
    if (!is_user_logged_in()) {
        return new WP_Error('rest_forbidden', __('REST API доступен только авторизованным.', 'wp-optimize-core'), ['status' => 401]);
    }
    return $result;
}

function wpoc_reduce_heartbeat_interval($settings) {
    $settings['interval'] = 120;
    return $settings;
}

function wpoc_clean_wp_head() {
    remove_action('wp_head', 'wp_generator');
    remove_action('wp_head', 'feed_links', 2);
    remove_action('wp_head', 'feed_links_extra', 3);
    remove_action('wp_head', 'rsd_link');
    remove_action('wp_head', 'wlwmanifest_link');
    remove_action('wp_head', 'index_rel_link');
    remove_action('wp_head', 'parent_post_rel_link', 10, 0);
    remove_action('wp_head', 'start_post_rel_link', 10, 0);
    remove_action('wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0);
    remove_action('wp_head', 'wp_shortlink_wp_head', 10, 0);
    remove_action('wp_head', 'wp_resource_hints', 2);
    remove_action('wp_head', 'rest_output_link_wp_head', 10);
    remove_action('wp_head', 'wp_oembed_add_discovery_links');
    remove_filter('wp_resource_hints', 'wp_resource_hints');
}

function wpoc_dequeue_gutenberg_front() {
    if (!is_admin() && !wp_is_json_request()) {
        wp_dequeue_style('wp-block-library');
        wp_dequeue_style('wp-block-library-theme');
        wp_dequeue_style('wc-blocks-style');
    }
}

function wpoc_disable_self_pings(&$links) {
    $home = trailingslashit(get_option('home'));
    foreach ($links as $l => $link) {
        if (0 === strpos($link, $home)) {
            unset($links[$l]);
        }
    }
}

function wpoc_disable_rss() {
    remove_action('wp_head', 'feed_links', 2);
    remove_action('wp_head', 'feed_links_extra', 3);
    add_action('do_feed', 'wpoc_disable_feed', 1);
    add_action('do_feed_rdf', 'wpoc_disable_feed', 1);
    add_action('do_feed_rss', 'wpoc_disable_feed', 1);
    add_action('do_feed_rss2', 'wpoc_disable_feed', 1);
    add_action('do_feed_atom', 'wpoc_disable_feed', 1);
    add_action('do_feed_rss2_comments', 'wpoc_disable_feed', 1);
    add_action('do_feed_atom_comments', 'wpoc_disable_feed', 1);
}

function wpoc_disable_feed() {
    wp_redirect(home_url(), 301);
    exit;
}

function wpoc_disable_author_archives() {
    if (is_author()) {
        global $wp_query;
        $wp_query->set_404();
        status_header(404);
        nocache_headers();
        include get_404_template();
        exit;
    }
}

function wpoc_remove_jquery_migrate($scripts) {
    if (isset($scripts->registered['jquery'])) {
        $scripts->registered['jquery']->deps = array_diff(
            $scripts->registered['jquery']->deps,
            ['jquery-migrate']
        );
    }
}

function wpoc_remove_script_version($src) {
    if (strpos($src, 'ver=')) {
        $src = remove_query_arg('ver', $src);
    }
    return $src;
}

// === НОВАЯ функция ===
function wpoc_send_last_modified_header() {
    if (is_singular()) {
        $post = get_queried_object();
        if ($post && in_array($post->post_type, get_post_types(['public' => true]), true)) {
            $last_modified = $post->post_modified_gmt;
        }
    } elseif (is_archive() || is_home()) {
        $posts = get_posts([
            'post_type'           => get_post_types(['public' => true]),
            'posts_per_page'      => 1,
            'post_status'         => 'publish',
            'orderby'             => 'modified',
            'suppress_filters'    => false,
        ]);
        if (!empty($posts[0])) {
            $last_modified = $posts[0]->post_modified_gmt;
        }
    } else {
        $last_modified = false;
    }

    if (!empty($last_modified)) {
        $last_modified_gmt = gmdate('D, d M Y H:i:s', strtotime($last_modified)) . ' GMT';
        if (!headers_sent()) {
            header('Last-Modified: ' . $last_modified_gmt);
        }
    }
}

// --- Экспорт / Импорт ---
add_action('admin_init', 'wpoc_handle_export_import');

function wpoc_handle_export_import() {
    if (isset($_GET['wpoc_action']) && $_GET['wpoc_action'] === 'export') {
        if (!wp_verify_nonce($_GET['_wpnonce'], 'wpoc_export')) {
            wp_die('Security check failed');
        }
        if (!current_user_can('manage_options')) {
            wp_die('Access denied');
        }

        $settings = get_option('wpoc_settings', wpoc_get_default_settings());
        $export_data = [
            'plugin' => 'WP Optimize Core',
            'version' => '1.4',
            'exported_at' => gmdate('c'),
            'site_url' => site_url(),
            'settings' => $settings,
        ];

        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="wp-optimize-core-settings-' . gmdate('Y-m-d') . '.json"');
        header('Cache-Control: no-store, no-cache, must-revalidate');
        echo wp_json_encode($export_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }

    if (isset($_POST['wpoc_import']) && current_user_can('manage_options')) {
        check_admin_referer('wpoc_import');

        $json_raw = wp_unslash($_POST['import_json'] ?? '');
        if (empty($json_raw)) {
            add_settings_error('wpoc_settings', 'import_empty', __('JSON не должен быть пустым.', 'wp-optimize-core'), 'error');
            return;
        }

        $data = json_decode($json_raw, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            add_settings_error('wpoc_settings', 'import_json_error', sprintf(__('Некорректный JSON: %s', 'wp-optimize-core'), json_last_error_msg()), 'error');
            return;
        }

        if (!isset($data['settings']) || !is_array($data['settings'])) {
            add_settings_error('wpoc_settings', 'import_format', __('Неверный формат: ключ "settings" отсутствует или не является массивом.', 'wp-optimize-core'), 'error');
            return;
        }

        $valid_keys = array_keys(wpoc_get_default_settings());
        $imported = [];
        foreach ($valid_keys as $key) {
            $imported[$key] = !empty($data['settings'][$key]) ? '1' : '0';
        }

        update_option('wpoc_settings', $imported);
        add_settings_error('wpoc_settings', 'import_success', __('Настройки успешно импортированы.', 'wp-optimize-core'), 'updated');
    }
}

// --- Диагностика ---
function wpoc_analyze_wp_head_safe() {
    ob_start();
    do_action('wp_head');
    $head_html = ob_get_clean();
    if ($head_html === false) $head_html = '';

    $script_count = preg_match_all('#<script#i', $head_html);
    $style_count  = preg_match_all('#<link[^>]+rel=["\']stylesheet["\']#i', $head_html) + preg_match_all('#<style#i', $head_html);
    $total_size   = strlen($head_html);

    $heavy_scripts = [];
    $heavy_styles  = [];

    if (preg_match_all('#<script[^>]+src=["\']([^"\']+)["\']#i', $head_html, $matches)) {
        foreach ($matches[1] as $src) {
            if (!str_starts_with($src, '') && !str_starts_with($src, '')) {
                $heavy_scripts[] = esc_url_raw($src);
            }
        }
    }

    if (preg_match_all('#<link[^>]+rel=["\']stylesheet["\'][^>]+href=["\']([^"\']+)["\']#i', $head_html, $matches)) {
        foreach ($matches[1] as $href) {
            if (!str_starts_with($href, '')) {
                $heavy_styles[] = esc_url_raw($href);
            }
        }
    }

    return [
        'script_count' => $script_count,
        'style_count'  => $style_count,
        'total_size'   => $total_size,
        'heavy_scripts' => array_unique(array_filter($heavy_scripts)),
        'heavy_styles'  => array_unique(array_filter($heavy_styles)),
    ];
}

function wpoc_analyze_autoloaded_options_safe() {
    global $wpdb;

    $rows = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT option_name, LENGTH(option_value) AS size
             FROM {$wpdb->options}
             WHERE autoload = %s
             ORDER BY size DESC",
            'yes'
        ),
        ARRAY_A
    );

    $total_size = 0;
    $top_10 = [];
    foreach ($rows as $i => $row) {
        $total_size += (int) $row['size'];
        if ($i < 10) {
            $top_10[] = $row;
        }
    }

    return [
        'total_count' => count($rows),
        'total_size'  => $total_size,
        'top_10'      => $top_10,
    ];
}

// --- Админка ---
add_action('admin_menu', 'wpoc_add_admin_menu');
add_action('admin_init', 'wpoc_register_settings');

function wpoc_add_admin_menu() {
    add_menu_page(
        __('Оптимизация WordPress', 'wp-optimize-core'),
        __('Оптимизация', 'wp-optimize-core'),
        'manage_options',
        'wp-optimize-core',
        'wpoc_settings_page',
        'dashicons-performance',
        81
    );
}

function wpoc_register_settings() {
    register_setting(
        'wpoc_options',
        'wpoc_settings',
        [
            'type'              => 'array',
            'sanitize_callback' => 'wpoc_sanitize_settings',
            'default'           => wpoc_get_default_settings(),
        ]
    );
}

function wpoc_get_default_settings() {
    return [
        // Основные
        'disable_emoji'               => '1',
        'disable_embed'               => '1',
        'disable_xmlrpc'              => '1',
        'disable_rest_for_guests'     => '0',
        'reduce_heartbeat'            => '1',
        'clean_wp_head'               => '1',
		'block_author_query_redirect' => '0',
        'disable_gutenberg_frontend'  => '1',
        'disable_self_pingbacks'      => '1',
        // Новые
        'disable_rss'                 => '0',
        'disable_author_archives'     => '0',
        'disable_jquery_migrate'      => '1',
        'hide_wp_version'             => '1',
		'disable_comment_website_field' => '0',
		'canonical_pagination_to_main' => '0',
        'disable_wpautop'             => '0',
        'disable_core_updates'        => '0',
        'block_sensitive_files'       => '0',
        // === НОВАЯ (16-я) ===
        'set_last_modified'           => '0',
    ];
}

function wpoc_sanitize_settings($input) {
    $valid = [];
    $keys = array_keys(wpoc_get_default_settings());
    foreach ($keys as $key) {
        $valid[$key] = !empty($input[$key]) ? '1' : '0';
    }
    return $valid;
}

// --- Страница настроек ---
function wpoc_settings_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    settings_errors('wpoc_settings');

    // Безопасное получение настроек с миграцией
    $saved_settings = get_option('wpoc_settings', []);
    $default_settings = wpoc_get_default_settings();
    foreach ($default_settings as $key => $default_value) {
        if (!isset($saved_settings[$key])) {
            $saved_settings[$key] = $default_value;
        }
    }
    $settings = $saved_settings;

    $current_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'settings';
    $tabs = [
        'settings' => __('Настройки'),
        'report'   => __('Отчёт'),
        'tools'    => __('Инструменты'),
    ];
    $active_tab = array_key_exists($current_tab, $tabs) ? $current_tab : 'settings';

    $head_report = null;
    $autoload_report = null;
    if ($active_tab === 'report') {
        try {
            $head_report = wpoc_analyze_wp_head_safe();
        } catch (Exception $e) {
            $head_report = ['error' => $e->getMessage()];
        }
        try {
            $autoload_report = wpoc_analyze_autoloaded_options_safe();
        } catch (Exception $e) {
            $autoload_report = ['error' => $e->getMessage()];
        }
    }
    ?>
    <div class="wrap">
        <h1><?= esc_html__('WP Optimize Core') ?></h1>
        <p><?= esc_html__('Плагин для безопасного отключения ненужных функций WordPress. Все изменения обратимы.') ?></p>

        <nav class="nav-tab-wrapper" aria-label="Secondary menu">
            <?php foreach ($tabs as $tab_id => $tab_name): ?>
                <a href="<?= esc_url(add_query_arg('tab', $tab_id)) ?>"
                   class="nav-tab <?= $active_tab === $tab_id ? 'nav-tab-active' : '' ?>">
                    <?= esc_html($tab_name) ?>
                </a>
            <?php endforeach; ?>
        </nav>

        <?php if ($active_tab === 'settings'): ?>
            <form method="post" action="options.php" style="margin-top: 20px;">
                <?php settings_fields('wpoc_options'); ?>
                <table class="form-table" role="presentation">
                    <!-- Эмодзи -->
                    <tr>
                        <th scope="row"><?= esc_html__('🎨 Эмодзи') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_emoji]" value="1" <?= checked($settings['disable_emoji'], '1', false) ?>>
                                <?= esc_html__('Отключить поддержку эмодзи') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Убирает скрипт wp-emoji-release.min.js и стили из <head> и админки.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Лишние 2 HTTP-запроса и ~5 КБ JS/CSS даже если эмодзи не используются.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Эмодзи могут отображаться как квадратики в устаревших ОС/браузерах.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, почти всегда.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Embed -->
                    <tr>
                        <th scope="row"><?= esc_html__('▶️ Встраивание (oEmbed)') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_embed]" value="1" <?= checked($settings['disable_embed'], '1', false) ?>>
                                <?= esc_html__('Отключить автоматическое встраивание') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Отключает преобразование ссылок YouTube/Twitter в интерактивные блоки.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Снижает количество HTTP-запросов и убирает внешние скрипты.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Ссылки останутся обычными — используйте блок «Встраивание» в редакторе.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если не вставляете ссылки напрямую в контент.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- XML-RPC -->
                    <tr>
                        <th scope="row"><?= esc_html__('⚠️ XML-RPC') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_xmlrpc]" value="1" <?= checked($settings['disable_xmlrpc'], '1', false) ?>>
                                <?= esc_html__('Полностью отключить XML-RPC') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Блокирует /xmlrpc.php — старый API удалённого управления.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Частая цель брутфорс-атак; создаёт нагрузку даже при неиспользовании.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Перестанут работать мобильные приложения WordPress и Pingbacks.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, в 99% случаев.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- REST API -->
                    <tr>
                        <th scope="row"><?= esc_html__('📡 REST API (для гостей)') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_rest_for_guests]" value="1" <?= checked($settings['disable_rest_for_guests'], '1', false) ?>>
                                <?= esc_html__('Запретить REST API неавторизованным') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Возвращает 401 ошибку при доступе к /wp-json/ без логина.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('REST API может использоваться для сбора данных и атак.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Не будут работать фронтенд-виджеты на базе REST API без авторизации.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если сайт не headless.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Heartbeat -->
                    <tr>
                        <th scope="row"><?= esc_html__('❤️ Heartbeat API') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[reduce_heartbeat]" value="1" <?= checked($settings['reduce_heartbeat'], '1', false) ?>>
                                <?= esc_html__('Снизить частоту до 120 секунд') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Меняет интервал фоновых запросов с 15–60 до 120 секунд.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Heartbeat часто «стучит» впустую, создавая нагрузку на сервер.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Уведомления о редактировании поста могут приходить с задержкой.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, особенно на shared/VPS хостингах.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- wp_head -->
                    <tr>
                        <th scope="row"><?= esc_html__('🧹 Очистка <head>') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[clean_wp_head]" value="1" <?= checked($settings['clean_wp_head'], '1', false) ?>>
                                <?= esc_html__('Удалить ненужные метатеги и ссылки') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что убирается:') ?></strong>
                                <ul style="margin: 5px 0;">
                                    <li><?= esc_html__('<meta name="generator"> — версия WordPress') ?></li>
                                    <li><?= esc_html__('RSD, WLWManifest — для старых редакторов') ?></li>
                                    <li><?= esc_html__('shortlink, adjacent_posts — редко используются') ?></li>
                                    <li><?= esc_html__('resource hints (dns-prefetch)') ?></li>
                                </ul>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Уменьшение HTML, ускорение парсинга, безопасность.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, практически всегда.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Gutenberg -->
                    <tr>
                        <th scope="row"><?= esc_html__('🧱 Стили Gutenberg на сайте') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_gutenberg_frontend]" value="1" <?= checked($settings['disable_gutenberg_frontend'], '1', false) ?>>
                                <?= esc_html__('Отключить стили блоков на фронтенде') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Убирает wp-block-library.css с публичной части сайта.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Эти стили (~30–100 КБ) часто переопределяются темой.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Блоки могут потерять стили — проверьте после включения.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если ваша тема стилизует блоки (Astra, GeneratePress и др.).') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Self Pingbacks -->
                    <tr>
                        <th scope="row"><?= esc_html__('🔁 Самопинги') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_self_pingbacks]" value="1" <?= checked($settings['disable_self_pingbacks'], '1', false) ?>>
                                <?= esc_html__('Запретить уведомления о ссылках на свои посты') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Запрещает WordPress отправлять pingback при ссылке на другой свой пост.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Это создаёт мусор в комментариях и лишние запросы.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('В комментариях перестанут появляться «внутренние» уведомления.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, без рисков.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- RSS -->
                    <tr>
                        <th scope="row"><?= esc_html__('📡 RSS-ленты') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_rss]" value="1" <?= checked($settings['disable_rss'], '1', false) ?>>
                                <?= esc_html__('Полностью отключить RSS и Atom') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Блокирует /feed/, /comments/feed/ и убирает метатеги лент.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('RSS редко используется на корпоративных сайтах; экономит ресурсы и снижает поверхность атак.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Подписчики RSS не смогут получать обновления.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если у вас нет блога или рассылки через RSS.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Авторские архивы -->
                    <tr>
                        <th scope="row"><?= esc_html__('👥 Архивы авторов') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_author_archives]" value="1" <?= checked($settings['disable_author_archives'], '1', false) ?>>
                                <?= esc_html__('Отключить страницы /author/имя/') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Возвращает 404 при попытке открыть архив автора.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Скрывает структуру сайта и имена администраторов (безопасность).') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Пользователи не увидят список постов по автору.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если сайт не мультиавторский.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- jQuery Migrate -->
                    <tr>
                        <th scope="row"><?= esc_html__('🧩 jQuery Migrate') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_jquery_migrate]" value="1" <?= checked($settings['disable_jquery_migrate'], '1', false) ?>>
                                <?= esc_html__('Убрать jquery-migrate.min.js') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Удаляет скрипт совместимости с jQuery 1.x (устаревший).') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Экономит ~8 КБ и ускоряет загрузку jQuery.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Старые плагины (2015–2018 гг.) могут сломаться.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если вы не используете устаревшие плагины.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Скрыть версию -->
                    <tr>
                        <th scope="row"><?= esc_html__('🕶️ Скрыть версию WordPress') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[hide_wp_version]" value="1" <?= checked($settings['hide_wp_version'], '1', false) ?>>
                                <?= esc_html__('Убрать ver= из CSS/JS и метатег generator') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Убирает ?ver=6.6 из URL скриптов и <meta name="generator">.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Затрудняет определение версии WordPress (безопасность).') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Никаких — только плюсы.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, всегда.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- wpautop -->
                    <tr>
                        <th scope="row"><?= esc_html__('⏎ Автоформатирование (wpautop)') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_wpautop]" value="1" <?= checked($settings['disable_wpautop'], '1', false) ?>>
                                <?= esc_html__('Отключить автоматическую вставку <p> и <br>') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Отключает функцию wpautop(), которая «оборачивает» текст в <p>.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Позволяет точно контролировать HTML, избегать лишних тегов в блоках.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Текст в «обычном» редакторе может отображаться без переносов.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ⚠️ <?= esc_html__('Только если вы используете Gutenberg или ACF с контролем HTML.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Обновления -->
                    <tr>
                        <th scope="row"><?= esc_html__('🔄 Фоновые обновления') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[disable_core_updates]" value="1" <?= checked($settings['disable_core_updates'], '1', false) ?>>
                                <?= esc_html__('Отключить проверку обновлений ядра, тем, плагинов') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Запрещает WordPress делать HTTP-запросы к api.wordpress.org.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Снижает нагрузку, исключает утечки данных (например, домена в логах WP.org).') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Обновления нужно устанавливать вручную или через CI/CD.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, на продакшене с контролем версий.') ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Last-Modified -->
                    <tr>
                        <th scope="row"><?= esc_html__('📅 Last-Modified заголовок') ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wpoc_settings[set_last_modified]" value="1" <?= checked($settings['set_last_modified'], '1', false) ?>>
                                <?= esc_html__('Автоматически отправлять заголовок Last-Modified') ?>
                            </label>
                            <p class="description">
                                <strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Отправляет HTTP-заголовок Last-Modified с датой последнего изменения поста/страницы/архива.') ?><br>
                                <strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Помогает браузерам, CDN и прокси-серверам эффективнее кэшировать контент, улучшает TTFB.') ?><br>
                                <strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Никаких — только положительный эффект для кэширования.') ?><br>
                                <strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, особенно при использовании CDN (Cloudflare, KeyCDN и др.).') ?>
                            </p>
                        </td>
                    </tr>
					
					<!-- Block Author Query Redirect -->
					<tr>
						<th scope="row"><?= esc_html__('🔒 Блокировка ?author=N') ?></th>
						<td>
							<label>
								<input type="checkbox" name="wpoc_settings[block_author_query_redirect]" value="1" <?= checked($settings['block_author_query_redirect'], '1', false) ?>>
								<?= esc_html__('Блокировать редирект с ?author=N на /author/имя/') ?>
							</label>
							<p class="description">
								<strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Запрещает WordPress редиректить ?author=1 на /author/имя/ — тем самым скрывает логин администратора.') ?><br>
								<strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Предотвращает раскрытие логина администратора при прямом запросе через GET-параметр.') ?><br>
								<strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('URL вида ?author=1 будут возвращать 404, а не редирект.') ?><br>
								<strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, особенно на сайтах без мультиавторства.') ?>
							</p>
						</td>
					</tr>
					
					
					<!-- Disable Comment Website Field -->
					<tr>
						<th scope="row"><?= esc_html__('💬 Поле "Сайт" в комментариях') ?></th>
						<td>
							<label>
								<input type="checkbox" name="wpoc_settings[disable_comment_website_field]" value="1" <?= checked($settings['disable_comment_website_field'], '1', false) ?>>
								<?= esc_html__('Скрыть поле "Сайт" в форме комментариев') ?>
							</label>
							<p class="description">
								<strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Убирает поле "Сайт" из формы комментариев и запрещает указывать URL в комментарии.') ?><br>
								<strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Снижает риск спама, упрощает форму, улучшает UX.') ?><br>
								<strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Пользователи не смогут указать сайт при комментировании.') ?><br>
								<strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, если вы не ожидаете, что пользователи будут указывать URL.') ?>
							</p>
						</td>
					</tr>
					
					<!-- Canonical Pagination -->
					<tr>
						<th scope="row"><?= esc_html__('🔗 Canonical на пагинации') ?></th>
						<td>
							<label>
								<input type="checkbox" name="wpoc_settings[canonical_pagination_to_main]" value="1" <?= checked($settings['canonical_pagination_to_main'], '1', false) ?>>
								<?= esc_html__('Использовать canonical на главную страницу архива, а не на /page/N') ?>
							</label>
							<p class="description">
								<strong><?= esc_html__('Что делает:') ?></strong> <?= esc_html__('Заменяет rel="canonical" на страницах пагинации (например /page/2/) на основной URL архива (например /category/имя/).') ?><br>
								<strong><?= esc_html__('Зачем:') ?></strong> <?= esc_html__('Предотвращает дубли страниц, концентрирует PageRank на главной странице архива.') ?><br>
								<strong><?= esc_html__('Последствия:') ?></strong> <?= esc_html__('Поисковики будут индексировать только первую страницу архива как основную.') ?><br>
								<strong><?= esc_html__('Рекомендуется:') ?></strong> ✅ <?= esc_html__('Да, особенно при использовании Yoast SEO или при высокой пагинации.') ?>
							</p>
						</td>
					</tr>

                </table>
                <?php submit_button(); ?>
            </form>
        <?php endif; ?>

        <?php if ($active_tab === 'report'): ?>
            <div style="margin-top: 20px;">
                <h2><?= esc_html__('📊 Отчёт о текущем состоянии') ?></h2>

                <?php if (isset($head_report['error'])): ?>
                    <div class="notice notice-error">
                        <p><strong><?= esc_html__('Ошибка при анализе <head>:') ?></strong> <?= esc_html($head_report['error']) ?></p>
                    </div>
                <?php else: ?>
                    <h3>1. <?= esc_html__('<head> секция') ?></h3>
                    <table class="wp-list-table widefat fixed striped">
                        <tr><th><?= esc_html__('Показатель') ?></th><th><?= esc_html__('Значение') ?></th></tr>
                        <tr><td><?= esc_html__('Скриптов') ?></td><td><?= (int) ($head_report['script_count'] ?? 0) ?></td></tr>
                        <tr><td><?= esc_html__('Стилей') ?></td><td><?= (int) ($head_report['style_count'] ?? 0) ?></td></tr>
                        <tr><td><?= esc_html__('Размер <head>') ?></td><td><?= $head_report['total_size'] ? size_format($head_report['total_size']) : '—' ?></td></tr>
                    </table>

                    <?php if (!empty($head_report['heavy_scripts'])): ?>
                        <h4><?= esc_html__('Внешние скрипты (могут блокировать рендеринг):') ?></h4>
                        <ul>
                            <?php foreach ($head_report['heavy_scripts'] as $src): ?>
                                <li><code><?= esc_html(basename($src)) ?></code> — <a href="<?= esc_url($src) ?>" target="_blank"><?= esc_html__('открыть') ?></a></li>
                            <?php endforeach; ?>
                        </ul>
                    <?php endif; ?>

                    <?php if (!empty($head_report['heavy_styles'])): ?>
                        <h4><?= esc_html__('Внешние таблицы стилей:') ?></h4>
                        <ul>
                            <?php foreach ($head_report['heavy_styles'] as $href): ?>
                                <li><code><?= esc_html(basename($href)) ?></code> — <a href="<?= esc_url($href) ?>" target="_blank"><?= esc_html__('открыть') ?></a></li>
                            <?php endforeach; ?>
                        </ul>
                    <?php endif; ?>
                <?php endif; ?>

                <?php if (isset($autoload_report['error'])): ?>
                    <div class="notice notice-error">
                        <p><strong><?= esc_html__('Ошибка при анализе автозагрузки:') ?></strong> <?= esc_html($autoload_report['error']) ?></p>
                    </div>
                <?php else: ?>
                    <h3>2. <?= esc_html__('Автозагружаемые опции') ?></h3>
                    <p><?= esc_html__('Эти опции загружаются при КАЖДОМ запросе к WordPress.') ?></p>
                    <table class="wp-list-table widefat fixed striped">
                        <tr><th><?= esc_html__('Показатель') ?></th><th><?= esc_html__('Значение') ?></th></tr>
                        <tr><td><?= esc_html__('Всего опций') ?></td><td><?= (int) ($autoload_report['total_count'] ?? 0) ?></td></tr>
                        <tr><td><?= esc_html__('Общий размер') ?></td><td><?= $autoload_report['total_size'] ? size_format($autoload_report['total_size']) : '—' ?></td></tr>
                    </table>

                    <?php if (!empty($autoload_report['top_10'])): ?>
                        <h4><?= esc_html__('Топ-10 самых тяжёлых опций:') ?></h4>
                        <table class="wp-list-table widefat fixed">
                            <thead><tr><th><?= esc_html__('Название') ?></th><th><?= esc_html__('Размер') ?></th></tr></thead>
                            <tbody>
                                <?php foreach ($autoload_report['top_10'] as $opt): ?>
                                    <tr>
                                        <td><code><?= esc_html($opt['option_name']) ?></code></td>
                                        <td><?= size_format($opt['size']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <p class="description">
                            <?= sprintf(
                                __('Совет: опции с префиксом %s можно отключить от автозагрузки SQL-запросом:', 'wp-optimize-core'),
                                '<code>_transient_</code>, <code>elementor_</code>, <code>woocommerce_</code>'
                            ) ?><br>
                            <code>UPDATE wp_options SET autoload = 'no' WHERE option_name = 'имя_опции';</code>
                        </p>
                    <?php else: ?>
                        <p class="description"><?= esc_html__('Нет автозагружаемых опций — отлично!') ?></p>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <?php if ($active_tab === 'tools'): ?>
            <div style="margin-top: 20px;">
                <h2><?= esc_html__('📤 Экспорт и импорт настроек') ?></h2>

                <h3><?= esc_html__('Экспорт текущих настроек') ?></h3>
                <p><?= esc_html__('Скачайте JSON-файл с вашими настройками для резервного копирования или переноса на другой сайт.') ?></p>
                <a href="<?= wp_nonce_url(
                    add_query_arg(['wpoc_action' => 'export']),
                    'wpoc_export'
                ) ?>" class="button button-primary">
                    📥 <?= esc_html__('Скачать настройки') ?>
                </a>

                <hr>

                <h3><?= esc_html__('Импорт настроек') ?></h3>
                <p><?= esc_html__('Вставьте содержимое ранее экспортированного JSON-файла:') ?></p>
                <form method="post">
                    <?php wp_nonce_field('wpoc_import'); ?>
                    <textarea name="import_json" rows="8" cols="80" placeholder='{"settings": {...}}' style="width: 100%; font-family: monospace;"></textarea><br><br>
                    <?php submit_button(
                        __('📤 Импортировать'),
                        'secondary',
                        'wpoc_import'
                    ); ?>
                </form>

                <hr>

                <h3><?= esc_html__('🛡️ Защита чувствительных файлов') ?></h3>
                <p><?= esc_html__('Добавьте эти правила на ваш сервер, чтобы запретить доступ к конфиденциальным файлам.') ?></p>

                <?php if (!empty($settings['block_sensitive_files'])): ?>
                    <div class="notice notice-warning">
                        <p><strong><?= esc_html__('✅ Опция «Защита чувствительных файлов» включена.') ?></strong><br>
                        <?= esc_html__('Убедитесь, что вы добавили правила ниже в конфигурацию сервера.') ?>
                        </p>
                    </div>
                <?php endif; ?>

                <h4>Apache (.htaccess)</h4>
                <pre style="background:#f5f5f5;padding:10px;border:1px solid #ddd;"><code># Запретить доступ к wp-config.php и .htaccess
&lt;Files wp-config.php&gt;
  Require all denied
&lt;/Files&gt;

&lt;Files .htaccess&gt;
  Require all denied
&lt;/Files&gt;</code></pre>

                <h4>Nginx</h4>
                <pre style="background:#f5f5f5;padding:10px;border:1px solid #ddd;"><code>location = /wp-config.php {
    deny all;
    return 404;
}
location = /.htaccess {
    deny all;
    return 404;
}</code></pre>
            </div>
        <?php endif; ?>

        <hr>
        <details>
            <summary><strong>ℹ️ <?= esc_html__('Важно: что НЕ делает этот плагин') ?></strong></summary>
            <ul>
                <li>❌ <?= esc_html__('Не кэширует страницы') ?></li>
                <li>❌ <?= esc_html__('Не сжимает изображения') ?></li>
                <li>❌ <?= esc_html__('Не оптимизирует базу данных') ?></li>
                <li>❌ <?= esc_html__('Не настраивает сервер') ?></li>
            </ul>
            <p><?= esc_html__('Этот плагин работает ТОЛЬКО с ядром WordPress — убирает то, что грузится «из коробки», но редко нужно.') ?></p>
        </details>
    </div>
    <?php
}

// Активация
register_activation_hook(__FILE__, 'wpoc_activate');
function wpoc_activate() {
    if (false === get_option('wpoc_settings')) {
        update_option('wpoc_settings', wpoc_get_default_settings());
    }
}