<?php
/**
 * Plugin Name: Simple Manual Redirects Pro
 * Plugin URI:  https://example.com
 * Description: PRO-версия: ручные 301-редиректы + лог 404 + уведомления в Telegram + массовое управление уведомлениями.
 * Version: 1.6.3
 * Author: You
 * License: GPL-2.0+
 * Text Domain: smr-pro
 */

if (!defined('ABSPATH')) exit;

class Simple_Manual_Redirects_Pro {

    private $opt_redirects   = 'smr_pro_redirects';
    private $opt_404_log     = 'smr_pro_404_log';
    private $opt_notified    = 'smr_pro_notified_log';
    private $opt_alerts_log  = 'smr_pro_alerts_log';
    private $opt_settings    = 'smr_pro_settings';

    private $log_max_size    = 500;

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'activate']);
        add_action('admin_menu', [$this, 'add_admin_page']);
        add_action('template_redirect', [$this, 'handle_404'], 0);
        add_action('admin_init', [$this, 'handle_forms']);
        add_action('smr_pro_cleanup_cron', [$this, 'cleanup_notified_log']);
        add_action('wp_dashboard_setup', [$this, 'add_dashboard_widget']);
        add_action('admin_head', [$this, 'dashboard_widget_styles']);
        
        if (!wp_next_scheduled('smr_pro_cleanup_cron')) {
            wp_schedule_event(time(), 'daily', 'smr_pro_cleanup_cron');
        }
    }

    public function activate() {
        $defaults = [
            'threshold_count'    => 50,
            'threshold_hours'    => 24,
            'repeat_delay_days'  => 7,
            'telegram_token'     => '',
            'telegram_chat_id'   => '',
        ];
        add_option($this->opt_settings, $defaults, '', 'no');
    }

    public function handle_404() {
        if (!is_404()) return;

        $redirects = get_option($this->opt_redirects, []);
        $log       = get_option($this->opt_404_log, []);
        $notified  = get_option($this->opt_notified, []);
        $settings  = get_option($this->opt_settings, []);

        $path = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
        $url = ltrim($path, '/');
        if ($url !== '' && substr($url, -1) === '/') {
            $url = substr($url, 0, -1);
        }

        if ($url === '' || preg_match('#^(wp-|xmlrpc|favicon|robots|apple-touch)#i', $url)) {
            return;
        }

        $url = preg_replace('#[^a-zA-Z0-9\-_\.\/\?=&]#', '', $url);
        if ($url === '') return;

        foreach ($redirects as $source => $target) {
            $norm_src = rtrim($source, '/');
            if (strcasecmp($url, $norm_src) === 0) {
                $final_url = $target === '' ? home_url('/') : 
                            (preg_match('#^https?://#i', $target) ? $target : home_url($target));
                
                $cur = trailingslashit(home_url($url));
                $tgt = trailingslashit($final_url);
                if ($cur !== $tgt) {
                    wp_redirect($final_url, 301);
                    exit;
                }
                return;
            }
        }

        $now = time();
        if (!isset($log[$url])) {
            $log[$url] = ['count' => 0, 'first' => $now, 'window_start' => $now];
        }
        $log[$url]['count']++;
        $log[$url]['last'] = $now;

        $window_sec = $settings['threshold_hours'] * HOUR_IN_SECONDS;
        if ($now - $log[$url]['window_start'] > $window_sec) {
            $log[$url]['count'] = 1;
            $log[$url]['window_start'] = $now;
        }

        $should_notify = false;
        $min_delay_sec = $settings['repeat_delay_days'] * DAY_IN_SECONDS;
        if ($log[$url]['count'] >= $settings['threshold_count']) {
            if (!isset($notified[$url])) {
                $should_notify = true;
            } elseif (($now - $notified[$url]) >= $min_delay_sec) {
                $should_notify = true;
            }
        }

        if ($should_notify) {
            $notified[$url] = $now;
            update_option($this->opt_notified, $notified, false);
            $this->send_telegram_alert($url, $log[$url]['count'], $log[$url]['first'], $now);
        }

        if (count($log) > $this->log_max_size) {
            uasort($log, fn($a, $b) => $b['count'] - $a['count']);
            $log = array_slice($log, 0, $this->log_max_size, true);
        }

        update_option($this->opt_404_log, $log, false);
    }

    private function send_telegram_alert($url, $count, $first_ts, $alert_ts) {
        $settings = get_option($this->opt_settings, []);
        $token = trim($settings['telegram_token'] ?? '');
        $chat_id = trim($settings['telegram_chat_id'] ?? '');

        if (!$token || !$chat_id) {
            $this->log_alert($url, $count, $first_ts, $alert_ts, false, 'telegram');
            return false;
        }

        $site = esc_html(get_bloginfo('name'));
        $home = esc_url_raw(home_url());
        $admin_url = esc_url_raw(admin_url('tools.php?page=smr-pro-settings&tab=redirects'));

        $message = sprintf(
            "⚠️ <b>Частые 404-ошибки на сайте</b>\n\n" .
            "🌐 <b>Сайт:</b> <a href=\"%s\">%s</a>\n" .
            "🔗 <b>URL:</b> <code>/%s</code>\n" .
            "📈 <b>Запросов:</b> <b>%d</b> за %d ч.\n" .
            "🕐 <b>Первый:</b> %s\n" .
            "🕓 <b>Последний:</b> %s\n\n" .
            "💡 <i>Рекомендуется добавить редирект в</i> <a href=\"%s\">админке</a>",
            $home,
            $site,
            esc_html($url),
            $count,
            $settings['threshold_hours'],
            date_i18n('Y-m-d H:i:s', $first_ts),
            date_i18n('Y-m-d H:i:s', $alert_ts),
            $admin_url
        );

        $api_url = 'https://api.telegram.org/bot' . $token . '/sendMessage';
        $data = [
            'chat_id'    => $chat_id,
            'text'       => $message,
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true,
        ];

        $response = wp_remote_post($api_url, [
            'body'      => $data,
            'timeout'   => 10,
            'sslverify' => true,
        ]);

        $success = !is_wp_error($response) && wp_remote_retrieve_response_code($response) == 200;
        $this->log_alert($url, $count, $first_ts, $alert_ts, $success, 'telegram');
        return $success;
    }

    private function log_alert($url, $count, $first_ts, $alert_ts, $success = true, $type = 'telegram') {
        $alerts = get_option($this->opt_alerts_log, []);
        $id = 'alert_' . substr(uniqid(), -8) . '_' . time();

        $alerts[$id] = [
            'id'           => $id,
            'url'          => $url,
            'count'        => $count,
            'first_ts'     => $first_ts,
            'alert_ts'     => $alert_ts,
            'type'         => $type,
            'status'       => $success ? 'sent' : 'failed',
            'is_handled'   => false,
            'dismissed'    => false,
        ];

        uasort($alerts, fn($a, $b) => $b['alert_ts'] - $a['alert_ts']);
        if (count($alerts) > 200) {
            $alerts = array_slice($alerts, 0, 200, true);
        }

        update_option($this->opt_alerts_log, $alerts, false);
    }

    public function cleanup_notified_log() {
        $notified = get_option($this->opt_notified, []);
        $now = time();
        $changed = false;

        foreach ($notified as $url => $ts) {
            if ($now - $ts > 90 * DAY_IN_SECONDS) {
                unset($notified[$url]);
                $changed = true;
            }
        }

        if ($changed) {
            update_option($this->opt_notified, $notified, false);
        }
    }

    public function add_dashboard_widget() {
        if (!current_user_can('manage_options')) return;
        wp_add_dashboard_widget(
            'smr_pro_dashboard_widget',
            '🚀 404-ошибки (Simple Redirects Pro)',
            [$this, 'render_dashboard_widget']
        );
    }

    public function dashboard_widget_styles() {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'dashboard') return;
        ?>
        <style>
            #smr_pro_dashboard_widget .widefat td {
                padding: 8px 10px;
            }
            #smr_pro_dashboard_widget .widefat code {
                background: #f5f5f5;
                padding: 2px 4px;
                border-radius: 3px;
                font-size: 0.9em;
            }
            #smr_pro_dashboard_widget .button-small {
                padding: 2px 8px;
                font-size: 12px;
                height: auto;
            }
            #smr_pro_dashboard_widget .notice-warning {
                margin: 0 -12px 12px;
            }
        </style>
        <?php
    }

    public function render_dashboard_widget() {
        $log = get_option($this->opt_404_log, []);
        $redirects = get_option($this->opt_redirects, []);
        $alerts = get_option($this->opt_alerts_log, []);

        $active_404 = array_filter($log, function($data, $url) use ($redirects) {
            return !isset($redirects[$url]);
        }, ARRAY_FILTER_USE_BOTH);

        uasort($active_404, fn($a, $b) => $b['count'] - $a['count']);

        $cutoff = time() - 7 * DAY_IN_SECONDS;
        $top = [];
        foreach ($active_404 as $url => $data) {
            if ($data['last'] >= $cutoff) {
                $top[$url] = $data;
                if (count($top) >= 3) break;
            }
        }

        $has_alerts = !empty($alerts);
        if (empty($top) && !$has_alerts): ?>
            <p>✅ Нет активных 404-ошибок за последние 7 дней.</p>
            <p class="description">
                <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=log')) ?>">Открыть полный лог</a>
            </p>
            <?php return;
        endif;

        if (!empty($top)): ?>
            <table class="widefat striped" style="margin-bottom:15px;">
                <thead>
                    <tr>
                        <th>URL</th>
                        <th style="text-align:right;">Запросов</th>
                        <th>Действие</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($top as $url => $data): ?>
                        <tr>
                            <td>
                                <code>/<?= esc_html($url) ?></code><br>
                                <small title="Последний запрос"><?= date_i18n('j M H:i', $data['last']) ?></small>
                            </td>
                            <td style="text-align:right;"><strong><?= (int)$data['count'] ?></strong></td>
                            <td>
                                <a href="<?= esc_url(add_query_arg([
                                    'page' => 'smr-pro-settings',
                                    'tab' => 'redirects',
                                    'source' => $url
                                ], admin_url('tools.php'))); ?>" 
                                   class="button button-primary button-small">
                                    → Редирект
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <p class="description" style="margin-top:10px;font-size:12px;">
            <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=log')) ?>">Полный лог 404</a> | 
            <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=alerts')) ?>">Все уведомления (<?= count($alerts) ?>)</a>
        </p>
    <?php
    }

    public function add_admin_page() {
        add_management_page(
            'Redirects Pro',
            'Redirects Pro',
            'manage_options',
            'smr-pro-settings',
            [$this, 'render_settings']
        );
    }

    public function handle_forms() {
        if (!current_user_can('manage_options')) return;

        // === ОЧИСТИТЬ ВСЁ УВЕДОМЛЕНИЙ ===
        if (isset($_POST['smr_pro_nonce']) && wp_verify_nonce($_POST['smr_pro_nonce'], 'smr_pro_action')) {
            if (isset($_POST['clear_all_alerts'])) {
                update_option($this->opt_alerts_log, [], false);
                wp_redirect(add_query_arg(['tab' => 'alerts'], admin_url('tools.php?page=smr-pro-settings')));
                exit;
            }
        }

        // === МАССОВЫЕ ДЕЙСТВИЯ ===
        if (isset($_POST['smr_pro_bulk_nonce']) && wp_verify_nonce($_POST['smr_pro_bulk_nonce'], 'smr_pro_bulk_action')) {
            $action = $_POST['action'] ?? $_POST['action2'] ?? '';
            $alert_ids = isset($_POST['alert_ids']) ? array_map('sanitize_text_field', (array) $_POST['alert_ids']) : [];

            if ($action === 'delete' && !empty($alert_ids)) {
                $alerts = get_option($this->opt_alerts_log, []);
                foreach ($alert_ids as $id) {
                    unset($alerts[$id]);
                }
                update_option($this->opt_alerts_log, $alerts, false);
                wp_redirect(add_query_arg(['tab' => 'alerts'], admin_url('tools.php?page=smr-pro-settings')));
                exit;
            }
        }

        // === ОСНОВНЫЕ ДЕЙСТВИЯ ===
        if (!isset($_POST['smr_pro_nonce']) || !wp_verify_nonce($_POST['smr_pro_nonce'], 'smr_pro_action')) return;

        $redirects = get_option($this->opt_redirects, []);
        $log       = get_option($this->opt_404_log, []);
        $alerts    = get_option($this->opt_alerts_log, []);
        $settings  = get_option($this->opt_settings, []);

        if (isset($_POST['save_settings'])) {
            $settings['threshold_count']   = max(1, intval($_POST['threshold_count'] ?? 50));
            $settings['threshold_hours']   = max(1, intval($_POST['threshold_hours'] ?? 24));
            $settings['repeat_delay_days'] = max(1, min(365, intval($_POST['repeat_delay_days'] ?? 7)));
            $settings['telegram_token']    = sanitize_text_field($_POST['telegram_token'] ?? '');
            $settings['telegram_chat_id']  = sanitize_text_field($_POST['telegram_chat_id'] ?? '');
            update_option($this->opt_settings, $settings, false);
        }

        if (isset($_POST['add_redirect'])) {
            $source = ltrim(trim($_POST['source'] ?? ''), '/');
            $target = trim($_POST['target'] ?? '');
            if ($source) {
                $target = ($target === '/' || $target === '') ? '' :
                         (preg_match('#^https?://#i', $target) ? $target : '/' . ltrim($target, '/'));
                $redirects[$source] = $target;

                foreach ($alerts as &$alert) {
                    if ($alert['url'] === $source && !$alert['dismissed']) {
                        $alert['is_handled'] = true;
                    }
                }
                unset($alert);
                update_option($this->opt_alerts_log, $alerts, false);

                unset($log[$source]);
                update_option($this->opt_404_log, $log, false);
            }
        }

        if (isset($_POST['delete_source'])) {
            unset($redirects[trim($_POST['delete_source'])]);
        }

        if (isset($_POST['clear_log'])) {
            $log = [];
            update_option($this->opt_404_log, $log, false);
        }

        if (isset($_POST['dismiss_alert'])) {
            $id = sanitize_text_field($_POST['dismiss_alert']);
            if (isset($alerts[$id])) {
                $alerts[$id]['dismissed'] = true;
                update_option($this->opt_alerts_log, $alerts, false);
            }
        }

        update_option($this->opt_redirects, $redirects, false);
    }

    public function render_settings() {
        $redirects = get_option($this->opt_redirects, []);
        $log       = get_option($this->opt_404_log, []);
        $alerts    = get_option($this->opt_alerts_log, []);
        $settings  = get_option($this->opt_settings, []);
        $tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'redirects';

        uasort($log, fn($a, $b) => $b['count'] - $a['count']);
        ?>
        <div class="wrap">
            <h1>🚀 Simple Manual Redirects <em>Pro</em></h1>

            <h2 class="nav-tab-wrapper">
                <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=redirects')) ?>" 
                   class="nav-tab <?= $tab === 'redirects' ? 'nav-tab-active' : '' ?>">Редиректы</a>
                <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=log')) ?>" 
                   class="nav-tab <?= $tab === 'log' ? 'nav-tab-active' : '' ?>">Лог 404</a>
                <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=alerts')) ?>" 
                   class="nav-tab <?= $tab === 'alerts' ? 'nav-tab-active' : '' ?>">
                    Уведомления (<?= count($alerts) ?>)
                </a>
                <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=settings')) ?>" 
                   class="nav-tab <?= $tab === 'settings' ? 'nav-tab-active' : '' ?>">Настройки</a>
            </h2>

            <?php if ($tab === 'redirects'): ?>
                <h2>➕ Добавить редирект</h2>
                <form method="post">
                    <?php wp_nonce_field('smr_pro_action', 'smr_pro_nonce'); ?>
                    <table class="form-table">
                        <tr><th>Исходный URL</th><td><input name="source" class="regular-text code" value="<?= isset($_GET['source']) ? esc_attr(sanitize_text_field($_GET['source'])) : '' ?>" required></td></tr>
                        <tr><th>Целевой URL</th><td><input name="target" class="regular-text code" placeholder="/, /new или https://..."></td></tr>
                    </table>
                    <?php submit_button('Добавить', 'primary', 'add_redirect'); ?>
                </form>

                <?php if ($redirects): ?>
                    <h2>📋 Редиректы</h2>
                    <table class="wp-list-table widefat striped">
                        <thead><tr><th>Исходный</th><th>Целевой</th><th>Действие</th></tr></thead>
                        <tbody>
                            <?php foreach ($redirects as $src => $tgt): ?>
                                <tr>
                                    <td><code><?= esc_html($src) ?></code></td>
                                    <td><?= $tgt === '' ? '<strong>Главная</strong>' : esc_html($tgt) ?></td>
                                    <td>
                                        <form method="post" style="display:inline;" onsubmit="return confirm('Удалить?')">
                                            <?php wp_nonce_field('smr_pro_action', 'smr_pro_nonce'); ?>
                                            <input type="hidden" name="delete_source" value="<?= esc_attr($src) ?>">
                                            <?php submit_button('Удалить', 'delete', null, false); ?>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>

            <?php elseif ($tab === 'log'): ?>
                <h2>📊 404-лог (<?= count($log) ?>)</h2>
                <?php if ($log): ?>
                    <form method="post" style="float:right;">
                        <?php wp_nonce_field('smr_pro_action', 'smr_pro_nonce'); ?>
                        <?php submit_button('🧹 Очистить', 'delete', 'clear_log', false); ?>
                    </form>
                    <table class="wp-list-table widefat striped">
                        <thead><tr><th>URL</th><th>Посещений</th><th>Первый</th><th>Последний</th><th>Действие</th></tr></thead>
                        <tbody>
                            <?php foreach ($log as $url => $d): ?>
                                <tr>
                                    <td><code>/<?= esc_html($url) ?></code></td>
                                    <td><strong><?= (int)$d['count'] ?></strong></td>
                                    <td><?= date_i18n('Y-m-d H:i', $d['first']) ?></td>
                                    <td><?= date_i18n('Y-m-d H:i', $d['last']) ?></td>
                                    <td><a href="<?= esc_url(add_query_arg(['tab' => 'redirects', 'source' => $url], admin_url('tools.php?page=smr-pro-settings'))) ?>" class="button">→ Редирект</a></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p>Лог пуст. <a href="<?= esc_url(home_url('/test-404-pro')) ?>" target="_blank">Протестировать</a></p>
                <?php endif; ?>

            <?php elseif ($tab === 'alerts'): ?>
                <h2>🔔 История уведомлений (<?= count($alerts) ?>)</h2>

                <?php if ($alerts): ?>
                    <!-- Форма: массовые действия -->
                    <form method="post">
                        <?php wp_nonce_field('smr_pro_bulk_action', 'smr_pro_bulk_nonce'); ?>
                        <div class="tablenav top">
                            <div class="alignleft actions bulkactions">
                                <label for="bulk-action-selector-top" class="screen-reader-text">Действия</label>
                                <select name="action" id="bulk-action-selector-top">
                                    <option value="-1">Действия</option>
                                    <option value="delete">Удалить</option>
                                </select>
                                <?php submit_button('Применить', 'action', '', false); ?>
                            </div>
                        </div>

                        <table class="wp-list-table widefat striped">
                            <thead>
                                <tr>
                                    <td class="manage-column column-cb check-column">
                                        <label class="screen-reader-text" for="cb-select-all-1">Выделить всё</label>
                                        <input id="cb-select-all-1" type="checkbox">
                                    </td>
                                    <th>Дата</th>
                                    <th>URL</th>
                                    <th>Запросов</th>
                                    <th>Тип</th>
                                    <th>Статус</th>
                                    <th>Действие</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                // Показываем ВСЕ уведомления — без фильтрации
                                foreach ($alerts as $a):
                                    $redirect_exists = isset($redirects[$a['url']]);
                                    $is_handled = $a['is_handled'] || $redirect_exists;
                                    $is_dismissed = $a['dismissed'];
                                    ?>
                                    <tr>
                                        <th scope="row" class="check-column">
                                            <label class="screen-reader-text" for="cb-select-<?php echo esc_attr($a['id']); ?>">Выбрать <?php echo esc_html($a['url']); ?></label>
                                            <input type="checkbox" name="alert_ids[]" value="<?php echo esc_attr($a['id']); ?>" id="cb-select-<?php echo esc_attr($a['id']); ?>">
                                        </th>
                                        <td><?= date_i18n('Y-m-d H:i', $a['alert_ts']) ?></td>
                                        <td><code>/<?= esc_html($a['url']) ?></code></td>
                                        <td><strong><?= (int)$a['count'] ?></strong></td>
                                        <td>
                                            <?php if ($a['type'] === 'telegram'): ?>
                                                <span style="color:#0088cc;">📱 Telegram</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($redirect_exists): ?>
                                                <span style="color:green;">✅ Обработано</span>
                                            <?php elseif ($is_dismissed): ?>
                                                <span style="color:#888;">➖ Игнорировано</span>
                                            <?php else: ?>
                                                <span style="color:#d63638;">⚠️ Требует внимания</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!$redirect_exists): ?>
                                                <a href="<?= esc_url(add_query_arg(['tab' => 'redirects', 'source' => $a['url']], admin_url('tools.php?page=smr-pro-settings'))) ?>" 
                                                   class="button button-primary button-small">→ Редирект</a>
                                            <?php endif; ?>
                                            
                                            <?php if (!$is_handled && !$is_dismissed): ?>
                                                <form method="post" style="display:inline;" onsubmit="return confirm('Игнорировать это уведомление?');">
                                                    <?php wp_nonce_field('smr_pro_action', 'smr_pro_nonce'); ?>
                                                    <input type="hidden" name="dismiss_alert" value="<?= esc_attr($a['id']) ?>">
                                                    <?php submit_button('Игнорировать', 'secondary button-small', null, false); ?>
                                                </form>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>

                        <div class="tablenav bottom">
                            <div class="alignleft actions bulkactions">
                                <label for="bulk-action-selector-bottom" class="screen-reader-text">Действия</label>
                                <select name="action2" id="bulk-action-selector-bottom">
                                    <option value="-1">Действия</option>
                                    <option value="delete">Удалить</option>
                                </select>
                                <?php submit_button('Применить', 'action', '', false); ?>
                            </div>
                        </div>
                    </form>

                    <!-- Форма: Очистить всё -->
                    <div style="margin-top:15px;">
                        <form method="post" onsubmit="return confirm('Очистить ВЕСЬ лог уведомлений? Это нельзя отменить.');">
                            <?php wp_nonce_field('smr_pro_action', 'smr_pro_nonce'); ?>
                            <input type="hidden" name="clear_all_alerts" value="1">
                            <?php submit_button('🧹 Очистить всё', 'delete', 'clear_all_btn', false); ?>
                        </form>
                    </div>
                <?php else: ?>
                    <p>Нет уведомлений. Настройте Telegram в <a href="<?= esc_url(admin_url('tools.php?page=smr-pro-settings&tab=settings')) ?>">настройках</a>.</p>
                <?php endif; ?>

            <?php elseif ($tab === 'settings'): ?>
                <h2>📡 Настройки уведомлений (Telegram)</h2>
                <p>
                    <strong>Как подключить:</strong>
                    <ol>
                        <li>Напишите <a href="https://t.me/BotFather" target="_blank">@BotFather</a> → <code>/newbot</code></li>
                        <li>Получите <strong>токен</strong> (вида <code>123456:ABC-DEF...</code>)</li>
                        <li>Напишите боту <code>/start</code></li>
                        <li>Узнайте <strong>Chat ID</strong> через <a href="https://t.me/userinfobot" target="_blank">@userinfobot</a></li>
                    </ol>
                </p>

                <form method="post">
                    <?php wp_nonce_field('smr_pro_action', 'smr_pro_nonce'); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Telegram Bot Token</th>
                            <td>
                                <input type="password" name="telegram_token" class="regular-text" 
                                       value="<?= esc_attr($settings['telegram_token'] ?? '') ?>"
                                       placeholder="123456:ABC-DEF1234ghIkl...">
                                <p class="description">
                                    Пример: <code>123456:ABC-DEF1234ghIkl-zyx57W2v1u123ew11</code>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Chat ID</th>
                            <td>
                                <input type="text" name="telegram_chat_id" class="regular-text" 
                                       value="<?= esc_attr($settings['telegram_chat_id'] ?? '') ?>"
                                       placeholder="123456789 или -1001234567890">
                                <p class="description">
                                    Ваш ID (положительный) или ID канала (отрицательный, начинается с <code>-100</code>)
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Порог срабатывания</th>
                            <td>
                                <label>
                                    <input type="number" name="threshold_count" min="1" value="<?= (int)$settings['threshold_count'] ?>" style="width:80px"> запросов
                                </label>
                                за
                                <label>
                                    <input type="number" name="threshold_hours" min="1" value="<?= (int)$settings['threshold_hours'] ?>" style="width:60px"> часов
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Повторное уведомление</th>
                            <td>
                                <label>
                                    Не ранее чем через
                                    <input type="number" name="repeat_delay_days" min="1" max="365" 
                                           value="<?= (int)($settings['repeat_delay_days'] ?? 7) ?>" 
                                           style="width:60px">
                                    дней
                                </label>
                            </td>
                        </tr>
                    </table>

                    <p class="submit">
                        <?php submit_button('Сохранить настройки', 'primary', 'save_settings', false); ?>
                        <?php submit_button('📤 Протестировать', 'secondary', 'test_telegram', false); ?>
                    </p>
                </form>

                <?php
                if (isset($_POST['test_telegram'])) {
                    $token = trim($_POST['telegram_token'] ?? '');
                    $chat_id = trim($_POST['telegram_chat_id'] ?? '');
                    if ($token && $chat_id) {
                        $test_msg = "✅ <b>Тестовое уведомление от Simple Redirects Pro</b>\n\n" .
                                    "Бот работает! Теперь вы будете получать оповещения о частых 404.";
                        
                        $res = wp_remote_post('https://api.telegram.org/bot' . $token . '/sendMessage', [
                            'body' => [
                                'chat_id' => $chat_id,
                                'text' => $test_msg,
                                'parse_mode' => 'HTML'
                            ],
                            'timeout' => 10
                        ]);
                        
                        if (!is_wp_error($res) && wp_remote_retrieve_response_code($res) == 200) {
                            echo '<div class="notice notice-success"><p>✅ Тестовое сообщение отправлено! Проверьте Telegram.</p></div>';
                        } else {
                            $err = is_wp_error($res) ? $res->get_error_message() : wp_remote_retrieve_body($res);
                            echo '<div class="notice notice-error"><p>❌ Ошибка: ' . esc_html($err) . '</p></div>';
                        }
                    } else {
                        echo '<div class="notice notice-warning"><p>⚠️ Заполните Token и Chat ID.</p></div>';
                    }
                }
                ?>
            <?php endif; ?>
        </div>
        <?php
    }
}

new Simple_Manual_Redirects_Pro();